/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.biometricprompt.nocipher;

import android.content.Context;
import android.content.DialogInterface;
import android.hardware.biometrics.BiometricPrompt;
import android.os.CancellationSignal;

public class BiometricAuthentication {
    private static final String TAG = "BioAuth";

    private BiometricPrompt mBiometricPrompt;
    private CancellationSignal mCancellationSignal;
    private Context mContext;

    // Process "Cancel" button
    private DialogInterface.OnClickListener cancelListener =
        new DialogInterface.OnClickListener () {
        @Override
        public void onClick(DialogInterface dialog, int which) {
            android.util.Log.d(TAG, "cancel");
            if (mCancellationSignal != null) {
                if (!mCancellationSignal.isCanceled())
                    mCancellationSignal.cancel();
            }
        }
    };

    public BiometricAuthentication(Context context) {
        mContext = context;
        BiometricPrompt.Builder builder = new BiometricPrompt.Builder(context);
        // Authentication prompt also provides a button for cacelling
        // Cancel is handled by DialogInterface.OnClickListener
        // given to setNegativeButton as the 3rd argument
        mBiometricPrompt = builder
            .setTitle("Please Authenticate")
            .setNegativeButton("Cancel",context.getMainExecutor() ,cancelListener)
            .build();
        reset();
    }

    public boolean startAuthentication(
            final BiometricPrompt.AuthenticationCallback callback) {

        mCancellationSignal = new CancellationSignal();

        // Callback which accepts the result of biometric authentication
        BiometricPrompt.AuthenticationCallback hook =
            new BiometricPrompt.AuthenticationCallback() {
                @Override
                public void onAuthenticationError(int errorCode,
                                                  CharSequence errString) {
                    android.util.Log.d(TAG, "onAuthenticationError");
                    if (callback != null) {
                        callback.onAuthenticationError(errorCode, errString);
                    }
                    reset();
                }

                @Override
                public void onAuthenticationHelp(int helpCode,
                                                 CharSequence helpString) {
                    android.util.Log.d(TAG, "onAuthenticationHelp");
                    if (callback != null) {
                        callback.onAuthenticationHelp(helpCode, helpString);
                    }
                }

                @Override
                public void onAuthenticationSucceeded(
                        BiometricPrompt.AuthenticationResult result) {
                    android.util.Log.d(TAG, "onAuthenticationSuccess");
                    if (callback != null) {
                        callback.onAuthenticationSucceeded(result);
                    }
                    reset();
                }

                @Override
                public void onAuthenticationFailed() {
                    android.util.Log.d(TAG, "onAuthenticationFailed");
                    if (callback != null) {
                        callback.onAuthenticationFailed();
                    }
                }
            };

        // Perform biomettic authentication
        // BiometricPrompt has a specific API for simple authentication
        // (not linked with key)
        android.util.Log.d(TAG, "Starting authentication");
        mBiometricPrompt.authenticate(mCancellationSignal,
                                      mContext.getMainExecutor(),
                                      hook);

        return true;
    }

    public boolean isAuthenticating() {
        return mCancellationSignal != null && !mCancellationSignal.isCanceled();
    }

    private void reset() {
        mCancellationSignal = null;
    }
}

